/*
#       lidsadm.c --- The Linux Intrusion Detection System Administration Tool 
#       (C) Huagang Xie 1999-2003 All rights reserved.
#       EMail complaints to xie@www.lids.org
#
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
*/
/* $Id: lidsadm.c,v 1.2.2.4 2005/03/28 05:42:56 xie Exp $ */

/* ------------------------------------------------------------------------- */

/* Includes */
#ifdef HAVE_CONFIG_H
#include <../config.h>
#endif

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>

#include "lidstools.h"
#include "lids_capflag.h"

/* These includes come from kernel (and override stdincs) */
#include "linux/capability.h"
#include "lidsif.h"

/* ------------------------------------------------------------------------- */

#ifdef DEBUG
#define LIDS_DBG(msg...)  printf( __FUNCTION__ ".l" LIDS_STR(__LINE__) ": " ##msg)
#else
#define LIDS_DBG(msg...)
#endif

void
exit_error(int status, const char *msg)
{
	fprintf(stderr, "lidsadm: %s\n", msg);
		perror("reason:");
	printf("\n");
	exit(status);
}

void
exit_version()
{
	printf("lidsadm version " VERSION " for the LIDS project\n");
	exit(1);
}

void
exit_normal()
{
	printf("lidsadm version " VERSION " for the LIDS project\n"
	       "Use 'lidsadm -h' for help\n");
	exit(1);
}

void
exit_help()
{
	entry_t *entry;

	printf("lidsadm version " VERSION " for the LIDS project\n"
	       "       Huagang Xie <xie[at]lids.org>\n"
	       "       Philippe Biondi <pbi[at]cartel-info.fr>\n\n"
	       "Usage: lidsadm -[S|I] -- [+|-][LIDS_FLAG] [...]\n"
#ifndef NOVIEW
	       "       lidsadm -V\n"
#endif
	       "       lidsadm -h\n\n"
	       "Commands:\n"
	       "       -S  To submit a password to switch some protections\n"
	       "       -I  To switch some protections without submitting password (sealing time)\n"
#ifndef NOVIEW
	       "       -V  To view current LIDS state (caps/flags)\n"
#endif
	       "       -v  To show the version\n"
	       "       -h  To list this help \n");

	printf("\nAvailable capabilities:\n");
	for_each_entry(cap_list, entry)
	    printf("%20s %s\n", entry->name, entry->desc);

	printf("\nAvailable flags:\n");
	for_each_entry(flag_list, entry)
	    printf("%20s %s\n", entry->name, entry->desc);

	exit(1);
}

void
lids_set_caps(int optind, int argc, char *argv[], lids_locks_t * locks)
{
	int i;

	for (i = optind; i < argc; i++) {
		entry_t *entry;
		int flag_entry;

		flag_entry = 0;
		entry = getentrybyname(cap_list, argv[i] + 1);
		if (!entry) {
			entry = getentrybyname(flag_list, argv[i] + 1);
			flag_entry = 1;
		}

		if (!entry) {
			fprintf(stderr, "    %s: invalid capability/flag\n",
				argv[i]);
			exit(1);
		}

		switch (argv[i][0]) {
		case '+':
			if (flag_entry)
				flag_raise(locks->flags, entry->val);
			else
				cap_raise(locks->cap_bset, entry->val);
			break;
		case '-':
			if (flag_entry)
				flag_lower(locks->flags, entry->val);
			else
				cap_lower(locks->cap_bset, entry->val);
			break;
		default:
			fprintf(stderr, "%s: invalid option\n", argv[i]);
			exit(1);
		}

	}
}

void
lids_switch(int optind, int argc, char *argv[])
{
	int lk;
	char passwd[BUFSIZ];
	lids_locks_t before, wanted, after;
	kernel_cap_t capchanges;

	if ((lk = open(LIDS_LOCKS, O_RDWR)) == -1) {
		perror("open");
		exit_error(2, "cannot open " LIDS_LOCKS);
	}
	if (read(lk, &before, sizeof (lids_locks_t)) == -1) {
		perror("read");
		exit_error(2, "cannot read " LIDS_LOCKS);
	}

//	wanted = before;
	memcpy(&wanted, &before, sizeof(lids_locks_t));
	lids_set_caps(optind, argc, argv, &wanted);

	wanted.magic1 = LIDS_MAGIC_1;
	wanted.magic2 = LIDS_MAGIC_2;
	wanted.magic3 = LIDS_MAGIC_3;
	wanted.magic4 = LIDS_MAGIC_4;

	if(read_crypto_passwd(passwd, 0, 1) < 0 ) {
		exit_error(2,"read password error");
	}

	memcpy(wanted.passwd, passwd, 32);
	if (write(lk, &wanted, sizeof (lids_locks_t)) == -1) {
		perror("write");
		exit_error(2, "cannot write " LIDS_LOCKS);
	}

	if (read(lk, &after, sizeof (lids_locks_t)) == -1) {
		perror("reread");
		exit_error(2, "cannot reread " LIDS_LOCKS);
	}

	close(lk);

	/*n
	 * Little warning to prevent people to loose too much time on this...
	 */
/*
	if (flag_raised(wanted.flags,getentrybyname(flag_list,"RELOAD_CONF")->val)) {
		printf("Don't forget to restart daemons for your changes to be effective.\n");
	}
*/

	capchanges = before.cap_bset ^ after.cap_bset;
	if (capchanges) {
		entry_t *cap;

		for_each_entry(cap_list, cap) {
			if (flag_raised(capchanges, cap->val)) {
				printf("-> %s is now %s\n",
				       cap->name,
				       flag_raised(after.cap_bset,
						   cap->
						   val) ? "allowed" :
				       "forbidden");
			}
		}
	} else {
		printf("No global capabilities have changed.\n");
	}

	/*
	 * Dont test RELOAD_CONF  because it is always read as 0
	 */

	flag_lower(wanted.flags, getentrybyname(flag_list, "RELOAD_CONF")->val);
	if ((wanted.flags != after.flags)) {
		fprintf(stderr, "Switching LIDS failed\n");
	}

/**/}
void
lids_init(int optind, int argc, char *argv[])
{
	int fd;
	lids_locks_t locks, locks2;
	locks.cap_bset = 0;
	locks.flags = 0;

	if ((fd = open(LIDS_LOCKS, O_RDWR)) == -1) {
		perror("open");
		exit_error(2, "cannot open " LIDS_LOCKS);
	}
	if (read(fd, &locks, sizeof (lids_locks_t)) == -1) {
		perror("read");
		exit_error(2, "cannot read " LIDS_LOCKS);
	}

	lids_set_caps(optind, argc, argv, &locks);
	flag_raise(locks.flags, LIDS_FLAGS_INIT);
	flag_raise(locks.flags, LIDS_FLAGS_LIDS_ON);

	locks.magic1 = LIDS_MAGIC_1;
	locks.magic2 = LIDS_MAGIC_2;
	locks.magic3 = LIDS_MAGIC_3;
	locks.magic4 = LIDS_MAGIC_4;
	locks.passwd[0] = 0;

	if (write(fd, &locks, sizeof (lids_locks_t)) == -1) {
		perror("write");
		exit_error(2, "cannot write " LIDS_LOCKS);
	}

	if (read(fd, &locks2, sizeof (lids_locks_t)) == -1) {
		perror("read");
		exit_error(2, "cannot reread " LIDS_LOCKS);
	}
	close(fd);

	/* 
	 * Dont test RELOAD_CONF test as 
	 * it is always read as 0
	 */
/*	flag_lower(locks.flags,getentrybyname(flag_list,"RELOAD_CONF")->val);*/

/* XXX Disabled until a better solution is implemented  */
#if 0
	/*   if ((locks.cap_bset != locks2.cap_bset) || */
	if ((locks.flags != locks2.flags))
		fprintf(stderr, "LIDS init failed\n");
#endif
}

#ifndef NOVIEW
void
lids_view()
{
	int fd;
	lids_locks_t locks;
	entry_t *entry;

	locks.cap_bset = 0;

	locks.flags = 0;

	if ((fd = open(LIDS_LOCKS, O_RDWR)) == -1) {
		perror("open");
		exit_error(2, "cannot open " LIDS_LOCKS);
	}
	if (read(fd, &locks, sizeof (lids_locks_t)) == -1) {
		perror("read");
		exit_error(2, "cannot read " LIDS_LOCKS);
	}
	close(fd);

	/* Maybe we need below option in next release
	 * if we want to disable to show lids parameter except LFS.
	 * OMO 
	if (lids_flag_raised(locks.flags, LIDS_FLAGS_LIDS_ON)
	 && lids_flag_raised(locks.flags, LIDS_FLAGS_LIDS_LOCAL_ON)
	 && !(lids_flag_raised(locks.flags, LIDS_FLAGS_ACL_DISCOVERY_ON))) {
		perror("open");
		exit_error(2, "Disable to check LIDS status when LIDS on\n");
	}
	*/

	for_each_entry(cap_list, entry)
	    printf("%30s %i\n", entry->name,
		   cap_raised(locks.cap_bset, entry->val) != 0);

	for_each_entry(flag_list, entry)
	    printf("%30s %i\n", entry->name,
		   cap_raised(locks.flags, entry->val) != 0);

}
#endif

#ifndef NOVIEW
static char shortopts[] = "VSIhv";
#else
static char shortopts[] = "SIhv";
#endif

int main(int argc, char **argv)
{
	int command = LIDS_NONE;
	int c;

	setentry(cap_list);
	setentry(flag_list);

	if (getuid() != 0) {
		exit_error(2, "You must be root to run this program");
	}

	while ((c = getopt(argc, argv, shortopts)) != -1) {
		switch (c) {
#ifndef NOVIEW
		case 'V':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_VIEW;
			break;
#endif
		case 'S':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_SWITCH;
			break;
		case 'I':
			if (command != LIDS_NONE)
				exit_error(2, "multiple commands specified");
			command = LIDS_INIT;
			break;
		case 'v':
			exit_version();
			break;
			break;
		case 'h':
			exit_help();
		default:
			break;
		}
	}

	if ((command != LIDS_INIT) && (command != LIDS_STATUS)
	    && (command != LIDS_SWITCH) && (optind < argc))
		exit_error(2, "unknown arguments found on commandline");
	if (command == LIDS_NONE || argc < 2)
		exit_normal();

	switch (command) {
	case LIDS_SWITCH:
		printf("SWITCH\n");
		lids_switch(optind, argc, argv);
		break;
	case LIDS_INIT:
		printf("INIT\n");
		lids_init(optind, argc, argv);
		break;
#ifndef NOVIEW
	case LIDS_VIEW:
		printf("VIEW\n");
		lids_view();
		break;
#endif
	}
	exit(0);
}
